require File.dirname(__FILE__) + '/../test_helper'

class VirtualDomainMockRequest
  attr_accessor :host
end

class VirtualDomainPageTest < Test::Unit::TestCase
  test_helper :page
  fixtures :pages, :page_parts

  def setup
    @request = VirtualDomainMockRequest.new
    Page.request = @request
    
    @page = pages(:homepage)
    @page.class_name = "VirtualDomainPage"
    @page.save
    @page = Page.find @page.id

    child_slug = "child"

    @child = @page.children.create :slug => child_slug, :title => "Child Page", :breadcrumb => child_slug, :status_id => Status["Published"].id

    @hostname_for_child = "hostname1"

    @config = { @hostname_for_child => child_slug, "hostname2" => "child2" }
  end

  def test_find_front_page_by_url
    set_virtual_domain_config "*" => @child.slug
    assert_equal @child, Page.find_by_url("/")
  end

  def test_find_subpage_by_url
    set_virtual_domain_config "*" => @child.slug

    subchild = @child.children.create :title => "Sub Child", :slug => "subchild", :breadcrumb => "breadcrumb", :status_id => Status["Published"].id

    assert_equal subchild, Page.find_by_url("/#{subchild.slug}")
  end

  def test_find_first_child_for_no_config_unknown_host
    assert_equal @page.children.find(:all).find { |child| child.published?}, Page.find_by_url("/")
  end

  def test_child_url
    set_virtual_domain_config
    assert_equal '/', @page.child_url(@child), "child url wasn't /"
  end

  def test_slug_for_host
    set_virtual_domain_config
    @config.each do |host, slug|
      assert_equal slug, @page.send(:slug_for_host, host)
    end
  end

  def test_child_for_host
    assert_child_for_host_config @child, @hostname_for_child, @config
  end

  def test_child_for_unknown_host
    assert_child_for_host_config @child, "random.host", "*" => @child.slug
  end

  def test_hostname
    host = "test.host"

    @request.host = host

    assert_equal @page.hostname, host, "behaviour hostname differs from request"

    @request.host = "something.else"

    assert_equal @page.hostname, host, "behaviour hostname changed after changing request hostname"
  end

  def test_hostname_with_no_request
    Page.request = nil
    assert_nil @page.hostname, "hostname wasnt nil when no request"
  end

  def test_sitemap
    set_virtual_domain_config
    assert_equal @config, @page.send(:site_map)
  end

  def test_default_sitemap
    default_map = @page.children.inject({}) { |memo, page| memo[page.slug] = page.slug; memo }
    assert_equal default_map, @page.send(:site_map)
  end

  def set_virtual_domain_config config_hash = @config, page = @page
    page.parts.create(:name => "config", :content => config_hash.to_yaml)
  end

  def assert_child_for_host_config child, hostname, config
    set_virtual_domain_config config
    assert_equal child, @page.send(:child_for_host, hostname)
  end

end
